<?php

namespace ltcms\route\rule;

use ltcms\route\Rule;
use ltcms\lib\Route;
use ltcms\lib\Request;
use Exception;

class RuleItem extends Rule
{
    /**
     * 是否为MISS规则
     * @var bool
     */
    protected $miss = false;

    /**
     * 是否为额外自动注册的OPTIONS规则
     * @var bool
     */
    protected $autoOption = false;

    /**
     * 检测路由（含路由匹配）
     */
    public function check(Request $request, $url, $completeMatch = false,$nums=1)
    {
        return $this->checkRule($request, $url, null, $completeMatch);
    }

    /**
     * 判断当前路由规则是否为自动注册的OPTIONS路由
     */
    public function isAutoOptions()
    {
        return $this->autoOption;
    }

    /**
     * 检测路由
     */
    public function checkRule(Request $request, $url, $match = null, $completeMatch = false)
    {
        // 检查参数有效性
        if (!$this->checkOption($this->option, $request)) {
            return false;
        }
        // 合并分组参数
        $option  = $this->getOption();
        $pattern = $this->getPattern();
        $url     = $this->urlSuffixCheck($request, $url, $option);
        if (is_null($match)) {
            $match = $this->checkMatch($url, $option, $pattern, $completeMatch);
        }
        if (false !== $match) {
            return $this->parseRule($request, $this->rule, $this->route, $url, $option, $match);
        }
        return false;
    }

    /**
     * 检测URL和规则路由是否匹配
     */
    private function checkMatch($url, array $option, array $pattern,$completeMatch)
    {
        if (isset($option['complete_match'])) {
            $completeMatch = $option['complete_match'];
        }
        $depr = $this->router->config('pathinfo_depr');

        $var  = [];
        $url  = $depr . str_replace('|', $depr, $url);
        $rule = $depr . str_replace('/', $depr, $this->rule);

        if ($depr == $rule && $depr != $url) {
            return false;
        }

        if (false === strpos($rule, '<')) {
            if (0 === strcasecmp($rule, $url) || 0 === strcasecmp($rule.$depr, $url) || (!$completeMatch && 0 === strncasecmp($rule . $depr, $url . $depr, strlen($rule . $depr)))) {
                return $var;
            }
            return false;
        }

        $slash = preg_quote('/-' . $depr, '/');

        if ($matchRule = preg_split('/[' . $slash . ']?<\w+\??>/', $rule, 2)) {
            if ($matchRule[0] && 0 !== strncasecmp($rule, $url, strlen($matchRule[0]))) {
                return false;
            }
        }
        if (preg_match_all('/[' . $slash . ']?<?\w+\??>?/', $rule, $matches)) {
            $regex = $this->buildRuleRegex($rule, $matches[0], $pattern, $option, $completeMatch);
            try {
                if (!preg_match('~^' . $regex . '~u', $url, $match)) {
                    return false;
                }
            } catch (\Exception $e) {
                throw new Exception('route pattern error');
            }
            foreach ($match as $key => $val) {
                if (is_string($key)) {
                    $var[$key] = $val;
                }
            }
        }
        // 成功匹配后返回URL中的动态变量数组
        return $var;
    }

    /**
     * URL后缀及Slash检查
     */
    protected function urlSuffixCheck(Request $request, $url, array $option = [])
    {
        // 是否区分 / 地址访问
        if (!empty($option['remove_slash']) && '/' != $this->rule) {
            $this->rule = rtrim($this->rule, '/');
            $url        = rtrim($url, '|');
        }
        if (isset($option['ext'])) {
            // 路由ext参数 优先于系统配置的URL伪静态后缀参数
            $url = preg_replace('/\.(' . $request->ext() . ')$/i', '', $url);
        }
        return $url;
    }

    /**
     * 设置当前路由为自动注册OPTIONS
     * @access public
     * @return $this
     */
    public function setAutoOptions()
    {
        $this->autoOption = true;
        return $this;
    }

    /**
     * 设置当前路由规则为MISS路由
     */
    public function setMiss()
    {
        $this->miss = true;
        return $this;
    }

    /**
     * 判断当前路由规则是否为MISS路由
     */
    public function isMiss()
    {
        return $this->miss;
    }

    /**
     * 架构函数
     */
    public function __construct(Route $router, RuleGroup $parent,$name = null,$rule = '', $route = null,$method = '*')
    {
        $this->router = $router;
        $this->parent = $parent;
        $this->name   = $name;
        $this->route  = $route;
        $this->method = $method;
        $this->setRule($rule);
        $this->router->setRule($this->rule, $this);
    }

    /**
     * 路由规则预处理
     * @access public
     * @param  string      $rule     路由规则
     * @return void
     */
    public function setRule($rule)
    {
        if ('$' == substr($rule, -1, 1)) {
            // 是否完整匹配
            $rule = substr($rule, 0, -1);
            $this->option['complete_match'] = true;
        }
        $rule = '/' != $rule ? ltrim($rule, '/') : '';
        if ($this->parent && $prefix = $this->parent->getFullName()) {
            $rule = $prefix . ($rule ? '/' . ltrim($rule, '/') : '');
        }
        if (false !== strpos($rule, ':')) {
            $this->rule = preg_replace(['/\[\:(\w+)\]/', '/\:(\w+)/'], ['<\1?>', '<\1>'], $rule);
        } else {
            $this->rule = $rule;
        }
        // 生成路由标识的快捷访问
        $this->setRuleName();
    }

    /**
     * 获取当前路由的URL后缀
     * @access public
     * @return string|null
     */
    public function getSuffix()
    {
        if (isset($this->option['ext'])) {
            $suffix = $this->option['ext'];
        } elseif ($this->parent->getOption('ext')) {
            $suffix = $this->parent->getOption('ext');
        } else {
            $suffix = null;
        }
        return $suffix;
    }

    /**
     * 设置别名
     */
    public function name($name)
    {
        $this->name = $name;
        $this->setRuleName(true);
        return $this;
    }

    /**
     * 设置路由标识 用于URL反解生成
     * @access protected
     * @param  bool $first 是否插入开头
     * @return void
     */
    protected function setRuleName($first = false)
    {
        if ($this->name) {
            $this->router->setName($this->name, $this, $first);
        }
    }
}