<?php

namespace ltcms\route\rule;

use ltcms\route\Rule;
use ltcms\lib\Request;
use ltcms\lib\Route;
use ltcms\container\Container;
use Exception;

class RuleGroup extends Rule
{
    /**
     * 分组路由（包括子分组）
     * @var array
     */
    protected $rules = [];

    /**
     * 分组路由规则
     * @var mixed
     */
    protected $rule;

    /**
     * MISS路由
     * @var RuleItem
     */
    protected $miss;

    /**
     * 完整名称
     * @var string
     */
    protected $fullName;

    /**
     * 分组别名
     * @var string
     */
    protected $alias;

    /**
     * 架构函数
     */
    public function __construct(Route $router, RuleGroup $parent = null, $name = '', $rule = null)
    {
        $this->router = $router;
        $this->parent = $parent;
        $this->rule   = $rule;
        $this->name   = trim($name, '/');

        $this->setFullName();

        if ($this->parent) {
            $this->parent->addRuleItem($this);
        }
    }

    /**
     * 设置分组的路由规则
     * @access public
     * @return void
     */
    protected function setFullName()
    {
        if (false !== strpos($this->name, ':')) {
            $this->name = preg_replace(['/\[\:(\w+)\]/', '/\:(\w+)/'], ['<\1?>', '<\1>'], $this->name);
        }
        if ($this->parent && $this->parent->getFullName()) {
            $this->fullName = $this->parent->getFullName() . ($this->name ? '/' . $this->name : '');
        } else {
            $this->fullName = $this->name;
        }
        if ($this->name) {
            $this->router->getRuleName()->setGroup($this->name, $this);
        }
    }

    /**
     * 获取完整分组Name
     */
    public function getFullName()
    {
        return $this->fullName ?: '';
    }

    /**
     * 分组URL匹配检查
     */
    protected function checkUrl($url)
    {
        if ($this->fullName) {
            $pos = strpos($this->fullName, '<');
            if (false !== $pos) {
                $str = substr($this->fullName, 0, $pos);
            } else {
                $str = $this->fullName;
            }
            if ($str && 0 !== stripos(str_replace('|', '/', $url), $str)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 检测分组路由
     */
    public function check(Request $request, $url, $completeMatch = false,$nums=1)
    {
        // 检查分组有效性
        if (!$this->checkOption($this->option, $request) || !$this->checkUrl($url)) {
            return false;
        }
        // 解析分组路由
        $this->parseGroupRule($this->rule);
        // 获取当前路由规则
        $method = strtolower($request->method());
        $rules  = $this->getRules($method);
        $option = $this->getOption();
        if (isset($option['complete_match'])) {
            $completeMatch = $option['complete_match'];
        }
        if (!empty($option['merge_rule_regex'])) {
            // 合并路由正则规则进行路由匹配检查
            $result = $this->checkMergeRuleRegex($request, $rules, $url, $completeMatch);
            if (false !== $result) {
                return $result;
            }
        }
        // 检查分组路由
        foreach ($rules as $key => $item) {
            $result = $item[1]->check($request, $url, $completeMatch,$nums+1);
            if (false !== $result) {
                return $result;
            }
        }
        if (!empty($option['dispatcher'])) {
            $result = $this->parseRule($request, '', $option['dispatcher'], $url, $option);
        } elseif ($this->miss && in_array($this->miss->getMethod(), ['*', $method])) {
            // 未匹配所有路由的路由规则处理
            $result = $this->parseRule($request, '', $this->miss->getRoute(), $url, $this->miss->getOption());
        } else {
            $result = false;
        }
        return $result;
    }

    /**
     * 获取分组的MISS路由
     */
    public function getMissRule()
    {
        return $this->miss;
    }

    /**
     * 注册MISS路由
     */
    public function miss($route, $method = '*')
    {
        // 创建路由规则实例
        $ruleItem = new RuleItem($this->router, $this, null, '', $route, strtolower($method));
        $ruleItem->setMiss();
        $this->miss = $ruleItem;
        return $ruleItem;
    }

    /**
     * 检测分组路由
     */
    protected function checkMergeRuleRegex(Request $request, array &$rules, $url, $completeMatch)
    {
        $depr  = $this->router->config('pathinfo_depr');
        $url   = $depr . str_replace('|', $depr, $url);
        $regex = [];
        $items = [];
        foreach ($rules as $key => $val) {
            $item = $val[1];
            if ($item instanceof RuleItem) {
                $rule = $depr . str_replace('/', $depr, $item->getRule());
                if ($depr == $rule && $depr != $url) {
                    unset($rules[$key]);
                    continue;
                }
                $complete = $item->getOption('complete_match', $completeMatch);
                if (false === strpos($rule, '<')) {
                    if (0 === strcasecmp($rule, $url) || 0 === strcasecmp($rule.$depr, $url) || (!$complete && 0 === strncasecmp($rule . $depr, $url . $depr, strlen($rule . $depr)))) {
                        return $item->checkRule($request, $url, []);
                    }
                    unset($rules[$key]);
                    continue;
                }
                $slash = preg_quote('/-' . $depr, '/');
                if ($matchRule = preg_split('/[' . $slash . ']?<\w+\??>/', $rule, 2)) {
                    if ($matchRule[0] && 0 !== strncasecmp($rule, $url, strlen($matchRule[0]))) {
                        unset($rules[$key]);
                        continue;
                    }
                }
                if (preg_match_all('/[' . $slash . ']?<?\w+\??>?/', $rule, $matches)) {
                    unset($rules[$key]);
                    $pattern = array_merge($this->getPattern(), $item->getPattern());
                    $option  = array_merge($this->getOption(), $item->getOption());
                    $regex[$key] = $this->buildRuleRegex($rule, $matches[0], $pattern, $option, $complete, '_LTCMS_' . $key);
                    $items[$key] = $item;
                }
            }
        }
        if (empty($regex)) {
            return false;
        }
        try {
            $result = preg_match('~^(?:' . implode('|', $regex) . ')~u', $url, $match);
        } catch (\Exception $e) {
            throw new Exception('route pattern error',1);
        }
        if ($result) {
            $var = [];
            foreach ($match as $key => $val) {
                if (is_string($key) && '' !== $val) {
                    list($name, $pos) = explode('_LTCMS_', $key);
                    $var[$name] = $val;
                }
            }
            if (!isset($pos)) {
                foreach ($regex as $key => $item) {
                    if (0 === strpos(str_replace(['\/', '\-', '\\' . $depr], ['/', '-', $depr], $item), $match[0])) {
                        $pos = $key;
                        break;
                    }
                }
            }
            $rule  = $items[$pos]->getRule();
            $array = $this->router->getRule($rule);
            foreach ($array as $item) {
                if (in_array($item->getMethod(), ['*', strtolower($request->method())])) {
                    $result = $item->checkRule($request, $url, $var);
                    if (false !== $result) {
                        return $result;
                    }
                }
            }
        }
        return false;
    }

    /**
     * 生成路由的正则规则
     */
    protected function buildRuleRegex($rule, array $match, array $pattern = [], array $option = [], $completeMatch = false, $suffix = '')
    {
        foreach ($match as $name) {
            $value = $this->buildNameRegex($name, $pattern, $suffix);
            if ($value) {
                $origin[]  = $name;
                $replace[] = $value;
            }
        }
        // 是否区分 / 地址访问
        if ('/' != $rule) {
            if (!empty($option['remove_slash'])) {
                $rule = rtrim($rule, '/');
            } elseif (substr($rule, -1) == '/') {
                $rule     = rtrim($rule, '/');
                $hasSlash = true;
            }
        }
        $regex = isset($replace) ? str_replace($origin, $replace, $rule) : $rule;
        $regex = str_replace([')?/', ')?-'], [')/', ')-'], $regex);
        if (isset($hasSlash)) {
            $regex .= '/';
        }
        return $regex . ($completeMatch ? '$' : '');
    }

    /**
     * 生成路由变量的正则规则
     */
    protected function buildNameRegex($name, array $pattern, $suffix)
    {
        $optional = '';
        $slash    = substr($name, 0, 1);
        if (in_array($slash, ['/', '-'])) {
            $prefix = $slash;
            $name   = substr($name, 1);
            $slash  = substr($name, 0, 1);
        } else {
            $prefix = '';
        }
        if ('<' != $slash) {
            return '';
        }
        if (strpos($name, '?')) {
            $name     = substr($name, 1, -2);
            $optional = '?';
        } elseif (strpos($name, '>')) {
            $name = substr($name, 1, -1);
        }
        if (isset($pattern[$name])) {
            $nameRule = $pattern[$name];
            if (0 === strpos($nameRule, '/') && '/' == substr($nameRule, -1)) {
                $nameRule = substr($nameRule, 1, -1);
            }
        } else {
            $nameRule = $this->router->config('default_route_pattern');
        }
        return '(' . $prefix . '(?<' . $name . $suffix . '>' . $nameRule . '))' . $optional;
    }

    /**
     * 获取分组的路由规则
     * @access public
     * @param  string $method 请求类型
     * @return array
     */
    public function getRules($method = '')
    {
        if ('' === $method) {
            return $this->rules;
        }
        return array_filter($this->rules, function ($item) use ($method) {
            return $method == $item[0] || '*' == $item[0];
        });
    }

    /**
     * 添加分组下的路由规则
     */
    public function addRule($rule, $route = null, $method = '*')
    {
        // 读取路由标识
        if (is_string($route)) {
            $name = $route;
        } else {
            $name = null;
        }
        $method = strtolower($method);
        if ('' === $rule || '/' === $rule) {
            $rule .= '$';
        }
        // 创建路由规则实例
        $ruleItem = new RuleItem($this->router, $this, $name, $rule, $route, $method);
        $this->addRuleItem($ruleItem, $method);
        return $ruleItem;
    }

    /**
     * 注册分组下的路由规则
     */
    public function addRuleItem(Rule $rule, $method = '*')
    {
        if (strpos($method, '|')) {
            $rule->method($method);
            $method = '*';
        }
        $this->rules[] = [$method, $rule];
        if ($rule instanceof RuleItem && 'options' != $method) {
            $this->rules[] = ['options', $rule->setAutoOptions()];
        }
        return $this;
    }


    /**
     * 延迟解析分组的路由规则
     * @access public
     * @param  bool $lazy 路由是否延迟解析
     * @return $this
     */
    public function lazy($lazy = true)
    {
        if (!$lazy) {
            $this->parseGroupRule($this->rule);
            $this->rule = null;
        }
        return $this;
    }

    /**
     * 合并分组的路由规则正则
     */
    public function mergeRuleRegex($merge = true)
    {
        return $this->setOption('merge_rule_regex', $merge);
    }

    /**
     * 解析分组和域名的路由规则及绑定
     */
    public function parseGroupRule($rule)
    {
        $origin = $this->router->getGroup();
        $this->router->setGroup($this);

        if ($rule instanceof \Closure) {
            Container::getInstance()->invokeFunction($rule);
        }
        $this->router->setGroup($origin);
    }

    /**
     * 清空分组下的路由规则
     */
    public function clear()
    {
        $this->rules = [];
    }

    public function __debugInfo()
    {
        return [
            'name'    => $this->name,
            'rule'    => $this->rule,
            'route'   => $this->route,
            'method'  => $this->method,
            'vars'    => $this->vars,
            'option'  => $this->option,
            'pattern' => $this->pattern,
            'rules' => $this->rules,
        ];
    }

}