<?php

namespace ltcms\route;

use Closure;
use ltcms\lib\Route;
use ltcms\lib\Request;
use ltcms\container\Container;
use ltcms\route\rule\RuleGroup;
use ltcms\route\dispatch\Callback as CallbackDispatch;
use ltcms\route\dispatch\Controller as ControllerDispatch;
use ltcms\route\dispatch\Url as UrlDispatch;

abstract class Rule
{
    /**
     * 路由标识
     * @var string
     */
    protected $name;

    /**
     * 所在域名
     * @var string
     */
    protected $domain;

    /**
     * 路由对象
     * @var Route
     */
    protected $router;

    /**
     * 路由所属分组
     * @var RuleGroup
     */
    protected $parent;

    /**
     * 路由规则
     * @var mixed
     */
    protected $rule;

    /**
     * 路由地址
     * @var string|Closure
     */
    protected $route;

    /**
     * 请求类型
     * @var string
     */
    protected $method;

    /**
     * 路由变量
     * @var array
     */
    protected $vars = [];

    /**
     * 路由参数
     * @var array
     */
    protected $option = [];

    /**
     * 路由变量规则
     * @var array
     */
    protected $pattern = [];

    /**
     * 需要和分组合并的路由参数
     * @var array
     */
    protected $mergeOptions = ['append', 'middleware'];

    /**
     * 获取路由参数
     * @access public
     * @param  string $name 变量名
     * @return mixed
     */
    public function config($name = '')
    {
        return $this->router->config($name);
    }

    abstract public function check(Request $request, $url, $completeMatch = false);

    /**
     * 指定路由中间件
     */
    public function middleware($middleware, ...$params)
    {
        if($middleware){
            if (empty($params) && is_array($middleware)) {
                $this->option['middleware'] = $middleware;
            } else {
                foreach ((array) $middleware as $item) {
                    $this->option['middleware'][] = [$item, $params];
                }
            }
        }
        return $this;
    }

    /**
     * 附加路由隐式参数
     */
    public function append(array $append = [])
    {
        $this->option['append'] = $append;
        return $this;
    }

    /**
     * 路由检查
     */
    protected function checkOption(array $option, Request $request)
    {
        // 检查当前路由是否匹配
        if (isset($option['match']) && is_callable($option['match'])) {
            if (false === $option['match']($this, $request)) {
                return false;
            }
        }

        // 请求类型检测
        if (!empty($option['method'])) {
            if (is_string($option['method']) && false === stripos($option['method'], $request->method())) {
                return false;
            }
        }

        // AJAX PJAX 请求检查
        foreach (['ajax', 'pjax', 'json'] as $item) {
            if (isset($option[$item])) {
                $call = 'is' . $item;
                if ($option[$item] && !$request->$call() || !$option[$item] && $request->$call()) {
                    return false;
                }
            }
        }

        // 伪静态后缀检测
        if ($request->url() != '/' && ((isset($option['ext']) && false === stripos('|' . $option['ext'] . '|', '|' . $request->ext() . '|')) || (isset($option['deny_ext']) && false !== stripos('|' . $option['deny_ext'] . '|', '|' . $request->ext() . '|')))) {
            return false;
        }

        // 域名检查
        if ((isset($option['domain']) && !in_array($option['domain'], [$request->host(true), $request->subDomain()]))) {
            return false;
        }

        // HTTPS检查
        if ((isset($option['https']) && $option['https'] && !$request->isSsl()) || (isset($option['https']) && !$option['https'] && $request->isSsl())) {
            return false;
        }

        // 请求参数检查
        if (isset($option['filter'])) {
            foreach ($option['filter'] as $name => $value) {
                if ($request->param($name, '', null) != $value) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 获取路由参数定义
     */
    public function getOption($name = '', $default = null)
    {
        $option = $this->option;
        if ($this->parent) {
            $parentOption = $this->parent->getOption();
            // 合并分组参数
            foreach ($this->mergeOptions as $item) {
                if (isset($parentOption[$item]) && isset($option[$item])) {
                    $option[$item] = array_merge($parentOption[$item], $option[$item]);
                }
            }
            $option = array_merge($parentOption, $option);
        }
        if ('' === $name) {
            return $option;
        }
        return $option[$name] ?? $default;
    }

    /**
     * 是否去除URL最后的斜线
     * @access public
     * @param  bool $remove 是否去除最后斜线
     * @return $this
     */
    public function removeSlash($remove = true)
    {
        return $this->setOption('remove_slash', $remove);
    }

    /**
     * 设置路由参数
     * @access public
     * @param  array $option 参数
     * @return $this
     */
    public function option(array $option)
    {
        $this->option = array_merge($this->option, $option);
        return $this;
    }

    /**
     * 设置单个路由参数
     * @access public
     * @param  string $name  参数名
     * @param  mixed  $value 值
     * @return $this
     */
    public function setOption($name, $value)
    {
        $this->option[$name] = $value;
        return $this;
    }

    /**
     * 注册变量规则
     * @access public
     * @param  array $pattern 变量规则
     * @return $this
     */
    public function pattern(array $pattern)
    {
        $this->pattern = array_merge($this->pattern, $pattern);
        return $this;
    }

    /**
     * 设置标识
     * @access public
     * @param  string $name 标识名
     * @return $this
     */
    public function name($name)
    {
        $this->name = $name;
        return $this;
    }


    /**
     * 获取路由对象
     * @access public
     * @return Route
     */
    public function getRouter()
    {
        return $this->router;
    }

    /**
     * 获取Name
     * @access public
     * @return string
     */
    public function getName()
    {
        return $this->name ?: '';
    }

    /**
     * 获取当前路由规则
     * @access public
     * @return mixed
     */
    public function getRule()
    {
        return $this->rule;
    }

    /**
     * 获取路由所在域名
     * @access public
     * @return string
     */
    public function getDomain()
    {
        if($this->domain){
            return $this->domain;
        }else{
            if($this->parent){
                return $this->parent->getDomain();
            }else{
                return "";
            }
        }
    }

    /**
     * 获取当前路由地址
     * @access public
     * @return mixed
     */
    public function getRoute()
    {
        return $this->route;
    }

    /**
     * 生成路由的正则规则
     */
    protected function buildRuleRegex($rule, array $match, array $pattern = [], array $option = [], $completeMatch = false, $suffix = '')
    {
        foreach ($match as $name) {
            $value = $this->buildNameRegex($name, $pattern, $suffix);
            if ($value) {
                $origin[]  = $name;
                $replace[] = $value;
            }
        }
        // 是否区分 / 地址访问
        if ('/' != $rule) {
            if (!empty($option['remove_slash'])) {
                $rule = rtrim($rule, '/');
            } elseif (substr($rule, -1) == '/') {
                $rule     = rtrim($rule, '/');
                $hasSlash = true;
            }
        }
        $regex = isset($replace) ? str_replace($origin, $replace, $rule) : $rule;
        $regex = str_replace([')?/', ')?-'], [')/', ')-'], $regex);
        if (isset($hasSlash)) {
            $regex .= '/';
        }
        return $regex . ($completeMatch ? '$' : '');
    }

    /**
     * 生成路由变量的正则规则
     */
    protected function buildNameRegex($name, array $pattern, $suffix)
    {
        $optional = '';
        $slash    = substr($name, 0, 1);
        if (in_array($slash, ['/', '-'])) {
            $prefix = $slash;
            $name   = substr($name, 1);
            $slash  = substr($name, 0, 1);
        } else {
            $prefix = '';
        }
        if ('<' != $slash) {
            return '';
        }
        if (strpos($name, '?')) {
            $name     = substr($name, 1, -2);
            $optional = '?';
        } elseif (strpos($name, '>')) {
            $name = substr($name, 1, -1);
        }
        if (isset($pattern[$name])) {
            $nameRule = $pattern[$name];
            if (0 === strpos($nameRule, '/') && '/' == substr($nameRule, -1)) {
                $nameRule = substr($nameRule, 1, -1);
            }
        } else {
            $nameRule = $this->router->config('default_route_pattern');
        }
        return '(' . $prefix . '(?<' . $name . $suffix . '>' . $nameRule . '))' . $optional;
    }

    /**
     * 获取变量规则定义
     */
    public function getPattern($name = '')
    {
        $pattern = $this->pattern;
        if ($this->parent) {
            $pattern = array_merge($this->parent->getPattern(), $pattern);
        }
        if ('' === $name) {
            return $pattern;
        }
        return $pattern[$name] ?? null;
    }

    /**
     * 获取当前路由的变量
     * @access public
     * @return array
     */
    public function getVars()
    {
        return $this->vars;
    }

    /**
     * 获取Parent对象
     * @access public
     * @return $this|null
     */
    public function getParent()
    {
        return $this->parent;
    }

    /**
     * 获取当前路由的请求类型
     * @access public
     * @return string
     */
    public function getMethod()
    {
        return strtolower($this->method);
    }

    /**
     * 设置路由请求类型
     * @access public
     * @param  string $method 请求类型
     * @return $this
     */
    public function method($method)
    {
        return $this->setOption('method', strtolower($method));
    }

    /**
     * 解析匹配到的规则路由
     */
    public function parseRule(Request $request, $rule, $route, $url, array $option = [], array $matches = [])
    {
        if (is_string($route) && isset($option['prefix'])) {
            // 路由地址前缀
            $route = $option['prefix'] . $route;
        }
        // 替换路由地址中的变量
        $extraParams = true;
        $search      = $replace      = [];
        $depr        = $this->router->config('pathinfo_depr');
        foreach ($matches as $key => $value) {
            $search[]  = '<' . $key . '>';
            $replace[] = $value;
            $search[]  = ':' . $key;
            $replace[] = $value;
            if (strpos($value, $depr)) {
                $extraParams = false;
            }
        }
        if (is_string($route)) {
            $route = str_replace($search, $replace, $route);
        }
        // 解析额外参数
        if ($extraParams) {
            $count = substr_count($rule, '/');
            $url   = array_slice(explode('|', $url), $count + 1);
            $this->parseUrlParams(implode('|', $url), $matches);
        }
        $this->vars = $matches;
        // 发起路由调度
        return $this->dispatch($request, $route, $option);
    }

    /**
     * 解析URL地址中的参数Request对象
     */
    protected function parseUrlParams($url, array &$var = [])
    {
        if ($url) {
            preg_replace_callback('/(\w+)\|([^\|]+)/', function ($match) use (&$var) {
                $var[$match[1]] = strip_tags($match[2]);
            }, $url);
        }
    }

    /**
     * 发起路由调度
     */
    protected function dispatch(Request $request, $route, array $option)
    {
        if (is_subclass_of($route, Dispatch::class)) {
            $result = new $route($request, $this, $route, $this->vars);
        } elseif ($route instanceof Closure) {
            // 执行闭包
            $result = new CallbackDispatch($request, $this, $route, $this->vars);
        } elseif (false !== strpos($route, '@') || false !== strpos($route, '::') || false !== strpos($route, '\\')) {
            // 路由到类的方法
            $route  = str_replace('::', '@', $route);
            $result = $this->dispatchMethod($request, $route);
        } else {
            // 路由到控制器/操作
            $result = $this->dispatchController($request, $route);
        }
        return $result;
    }

    /**
     * 解析URL的pathinfo参数
     */
    public function parseUrlPath($url)
    {
        // 分隔符替换 确保路由定义使用统一的分隔符
        $url = str_replace('|', '/', $url);
        $url = trim($url, '/');
        if (strpos($url, '/')) {
            // [控制器/操作]
            $path = explode('/', $url);
        } else {
            $path = [$url];
        }
        return $path;
    }

    /**
     * 解析URL地址为 模块/控制器/操作
     */
    protected function dispatchMethod(Request $request, $route)
    {
        $path = $this->parseUrlPath($route);
        $route  = str_replace('/', '@', implode('/', $path));
        $method = strpos($route, '@') ? explode('@', $route) : $route;
        return new CallbackDispatch($request, $this, $method, $this->vars);
    }

    /**
     * 解析URL地址为 模块/控制器/操作
     */
    protected function dispatchController(Request $request, $route)
    {
        $path = $this->parseUrlPath($route);
        $action     = array_pop($path);
        $controller = !empty($path) ? array_pop($path) : null;
        $namespace=!empty($path) ? implode("\\",$path) : null;
        if(strpos($route,"plugins")!==0){
            $namespace="";
        }
        // 路由到模块/控制器/操作
        return (new ControllerDispatch($request, $this, [$controller, $action], $this->vars))->setNamespace($namespace);
    }

    /**
     * 设置路由参数
     */
    public function __call($method, $args)
    {
        if (count($args) > 1) {
            $args[0] = $args;
        }
        array_unshift($args, $method);
        return call_user_func_array([$this, 'setOption'], $args);
    }

    public function __sleep()
    {
        return ['name', 'rule', 'route', 'method', 'vars', 'option', 'pattern'];
    }

    public function __wakeup()
    {
        $this->router = Container::pull('route');
    }

    public function __debugInfo()
    {
        return [
            'name'    => $this->name,
            'rule'    => $this->rule,
            'route'   => $this->route,
            'method'  => $this->method,
            'vars'    => $this->vars,
            'option'  => $this->option,
            'pattern' => $this->pattern,
        ];
    }
}