<?php

namespace ltcms\lib;

use ltcms\App;
use Closure;
use Exception;

class MultiApp
{
    protected $app;

    public function __construct(App $app)
    {
        $this->app  = $app;
    }

    /**
     * 多应用解析
     */
    public function handle($request, Closure $next)
    {
        if(!$this->parseMultiApp()){
            return $next($request);
        }
        return $this->app->middleware->pipeline('app')
            ->send($request)
            ->then(function ($request) use ($next) {
                return $next($request);
            });
    }

    /**
     * 获取当前运行入口名称
     */
    protected function getScriptName()
    {
        if (isset($_SERVER['SCRIPT_FILENAME'])) {
            $file = $_SERVER['SCRIPT_FILENAME'];
        } elseif (isset($_SERVER['argv'][0])) {
            $file = realpath($_SERVER['argv'][0]);
        }
        return isset($file) ? pathinfo($file, PATHINFO_FILENAME) : '';
    }

    /**
     * 解析多应用
     * @return bool
     */
    public function parseMultiApp()
    {
        $scriptName = $this->getScriptName();
        $defaultApp = $this->app->config->get('app.default_app') ?: 'index';
        $appName    = $this->app->http->getName();
        $express = $this->app->config->get('app.app_express', false);
        if ($appName || ($scriptName && !in_array($scriptName, ['index', 'ltcms']))) {
            $appName = $appName ?: $scriptName;
            $prefix="/".$scriptName.".php";
            $this->app->http->setBind();
        } else {
            // 自动多应用识别
            $this->app->http->setBind(false);
            $appName    = null;
            if (!$this->app->http->isBind()) {
                $path = $this->app->request->pathinfo();
                $map  = $this->app->config->get('app.app_map', []);
                $deny = $this->app->config->get('app.deny_app_list', []);
                $name = current(explode('/', $path));
                if (strpos($name, '.')) {
                    $name = strstr($name, '.', true);
                }
                if (isset($map[$name])) {
                    if ($map[$name] instanceof Closure) {
                        $result  = call_user_func_array($map[$name], [$this->app]);
                        $appName = $result ?: $name;
                    } else {
                        $appName = $map[$name];
                    }
                } elseif ($name && (false !== array_search($name, $map) || in_array($name, $deny))) {
                    throw new Exception('app not exists:' . $name,404);
                } elseif ($name && isset($map['*'])) {
                    $appName = $map['*'];
                } else {
                    $appName = $name ?: $defaultApp;
                    $appPath = $this->app->http->getPath() ?: $this->app->getAppPath() . $appName . DIRECTORY_SEPARATOR;
                    if (!is_dir($appPath)) {
                        if ($express) {
                            $prefix="";
                            if(!defined("LTCMS_ROUTE_PREFIX")){
                                define("LTCMS_ROUTE_PREFIX",$prefix);
                            }
                            $this->setApp($defaultApp);
                            return true;
                        } else {
                            return false;
                        }
                    }
                }
                if ($name) {
                    $this->app->request->setRoot('/' . $name);
                    $this->app->request->setPathinfo(strpos($path, '/') ? ltrim(strstr($path, '/'), '/') : '');
                }
                $prefix=$name ?"/".$name: "";
            }else{
                $prefix="";
            }
        }
        if(!defined("LTCMS_ROUTE_PREFIX")){
            define("LTCMS_ROUTE_PREFIX",$prefix);
        }
        if ($express) {
            $this->setApp($appName ?: $defaultApp);
            return true;
        }else{
            return false;
        }
    }

    /**
     * 获取路由目录
     * @access protected
     * @return string
     */
    protected function getRoutePath()
    {
        return $this->app->getAppPath() . 'route' . DIRECTORY_SEPARATOR;
    }

    /**
     * 设置应用
     * @param string $appName
     */
    protected function setApp($appName)
    {
        if($appName){
            $this->app->http->name($appName);
            $appPath = $this->app->http->getPath() ?: $this->app->getAppPath() . $appName . DIRECTORY_SEPARATOR;
            $this->app->setAppPath($appPath);
            // 设置应用命名空间
            $this->app->setNamespace($this->app->config->get('app.app_namespace') ?: 'app\\' . $appName);

            if (is_dir($appPath)) {
                $this->app->setRuntimePath($this->app->getRuntimePath() . $appName . DIRECTORY_SEPARATOR);
                $this->app->http->setRoutePath($this->getRoutePath());
                //加载应用
                if(!defined("LTCMS_APP_NAME")){
                    define("LTCMS_APP_NAME",$appName);
                }
                $this->loadApp($appName, $appPath);
            }
        }
    }

    /**
     * 加载应用文件
     * @param string $appName 应用名
     * @return void
     */
    protected function loadApp($appName, $appPath)
    {
        //加载助手函数
        $helpPath=$appPath."helper";
        Helper::loader(array($helpPath));

        //加载配置数据
        $files=[];
        $files[]= $appPath . 'config';
        $this->app->config->loader($files);

        //加载应用事件
        $event_file=$appPath."config".DIRECTORY_SEPARATOR."event".$this->app->getConfigExt();
        if (is_file($event_file)) {
            $even_data=$this->app->config->parse($event_file);
            $this->app->loadEvent($even_data);
        }

        /**
         * 加载中间件
         */
        if (is_file($appPath . 'middleware.php')) {
            $this->app->http->loadMiddleware('app');
        }

        /**
         * 加载容器
         */
        if (is_file($appPath . 'container.php')) {
            $this->app->bind(include $appPath . 'container.php');
        }

        // 加载应用默认语言包
        $this->app->loadLangPack();
    }

    /**
     * 打印信息控制
     * @return array
     */
    public function __debugInfo()
    {
        return [];
    }
}