<?php

namespace ltcms\lib;

use ArrayAccess;

/**
 * 获取环境变量
 */
class Env  implements ArrayAccess
{
    /**
     * 环境变量数据
     * @var array
     */
    protected $data = [];

    /**
     * 数据转换映射
     * @var array
     */
    protected $convert = [
        'true'  => true,
        'false' => false,
        'off'   => false,
        'on'    => true,
    ];

    public function __construct()
    {
        $this->data = $_ENV;
    }

    /**
     * 读取环境变量定义文件
     */
    public function load($file)
    {
        $env = parse_ini_file($file, true, INI_SCANNER_RAW) ?: [];
        $this->set($env);
    }

    /**
     * 获取环境变量值
     */
    public function get($name = null, $default = null)
    {
        if (is_null($name)) {
            return $this->data;
        }
        $name = strtoupper(str_replace('.', '_', $name));
        if (isset($this->data[$name])) {
            $result = $this->data[$name];
            if (is_string($result) && isset($this->convert[$result])) {
                return $this->convert[$result];
            }
            return $result;
        }
        return $this->getEnv($name, $default);
    }

    /**
     * 获取数据
     */
    protected function getEnv($name, $default = null)
    {
        $result = getenv('PHP_' . $name);
        if (false === $result) {
            return $default;
        }
        if ('false' === $result) {
            $result = false;
        } elseif ('true' === $result) {
            $result = true;
        }
        if (!isset($this->data[$name])) {
            $this->data[$name] = $result;
        }
        return $result;
    }

    /**
     * 设置环境变量值
     */
    public function set($env, $value = null)
    {
        if (is_array($env)) {
            $env = array_change_key_case($env, CASE_UPPER);
            foreach ($env as $key => $val) {
                if (is_array($val)) {
                    foreach ($val as $k => $v) {
                        $this->data[$key . '_' . strtoupper($k)] = $v;
                    }
                } else {
                    $this->data[$key] = $val;
                }
            }
        } else {
            $name = strtoupper(str_replace('.', '_', $env));
            $this->data[$name] = $value;
        }
    }

    /**
     * 检测是否存在环境变量
     * @access public
     * @param string $name 参数名
     * @return bool
     */
    public function has($name)
    {
        return !is_null($this->get($name));
    }

    /**
     * 设置环境变量
     * @access public
     * @param string $name  参数名
     * @param mixed  $value 值
     */
    public function __set($name, $value)
    {
        $this->set($name, $value);
    }

    /**
     * 获取环境变量
     * @access public
     * @param string $name 参数名
     * @return mixed
     */
    public function __get($name)
    {
        return $this->get($name);
    }

    /**
     * 检测是否存在环境变量
     * @access public
     * @param string $name 参数名
     * @return bool
     */
    public function __isset($name)
    {
        return $this->has($name);
    }

    public function offsetSet($name, $value)
    {
        $this->set($name, $value);
    }

    public function offsetExists($name)
    {
        return $this->__isset($name);
    }

    public function offsetUnset($name)
    {
        throw new \Exception('not support: unset');
    }

    public function offsetGet($name)
    {
        return $this->get($name);
    }
}