<?php

namespace ltcms\db\paginator\driver;

use ltcms\db\Paginator;

/**
 * Bootstrap 分页驱动
 */
class Bootstrap extends Paginator
{

    /**
     * 上一页按钮
     * @param  $text
     * @return
     */
    protected function getPreviousButton( $text = "&laquo;")
    {
        if ($this->currentPage() <= 1) {
            return $this->getDisabledTextWrapper($text);
        }
        $url = $this->url($this->currentPage() - 1);
        return $this->getPageLinkWrapper($url, $text);
    }

    /**
     * 下一页按钮
     * @param  $text
     * @return
     */
    protected function getNextButton( $text = '&raquo;')
    {
        if (!$this->hasMore) {
            return $this->getDisabledTextWrapper($text);
        }
        $url = $this->url($this->currentPage() + 1);
        return $this->getPageLinkWrapper($url, $text);
    }

    /**
     * 页码按钮
     * @return
     */
    protected function getLinks()
    {
        if ($this->simple) {
            return '';
        }
        $block = [
            'first'  => null,
            'slider' => null,
            'last'   => null,
        ];
        $side   = 3;
        $window = $side * 2;
        if ($this->totalPage < $window + 6) {
            $block['first'] = $this->getUrlRange(1, $this->totalPage);
        } elseif ($this->currentPage <= $window) {
            $block['first'] = $this->getUrlRange(1, $window + 2);
            $block['last']  = $this->getUrlRange($this->totalPage - 1, $this->totalPage);
        } elseif ($this->currentPage > ($this->totalPage - $window)) {
            $block['first'] = $this->getUrlRange(1, 2);
            $block['last']  = $this->getUrlRange($this->totalPage - ($window + 2), $this->totalPage);
        } else {
            $block['first']  = $this->getUrlRange(1, 2);
            $block['slider'] = $this->getUrlRange($this->currentPage - $side, $this->currentPage + $side);
            $block['last']   = $this->getUrlRange($this->totalPage - 1, $this->totalPage);
        }
        $html = '';
        if (is_array($block['first'])) {
            $html .= $this->getUrlLinks($block['first']);
        }
        if (is_array($block['slider'])) {
            $html .= $this->getDots();
            $html .= $this->getUrlLinks($block['slider']);
        }
        if (is_array($block['last'])) {
            $html .= $this->getDots();
            $html .= $this->getUrlLinks($block['last']);
        }
        return $html;
    }

    /**
     * 渲染分页html
     * @return mixed
     */
    public function render()
    {
        if ($this->hasPages()) {
            if ($this->simple) {
                return sprintf(
                    '<ul class="pager">%s %s</ul>',
                    $this->getPreviousButton(),
                    $this->getNextButton()
                );
            } else {
                return sprintf(
                    '<ul class="pagination">%s %s %s</ul>',
                    $this->getPreviousButton(),
                    $this->getLinks(),
                    $this->getNextButton()
                );
            }
        }
    }

    /**
     * 生成一个可点击的按钮
     *
     * @param   $url
     * @param   $page
     * @return
     */
    protected function getAvailablePageWrapper( $url,  $page)
    {
        return '<li><a href="' . htmlentities($url) . '">' . $page . '</a></li>';
    }

    /**
     * 生成一个禁用的按钮
     *
     * @param   $text
     * @return
     */
    protected function getDisabledTextWrapper( $text)
    {
        return '<li class="disabled"><span>' . $text . '</span></li>';
    }

    /**
     * 生成一个激活的按钮
     *
     * @param   $text
     * @return
     */
    protected function getActivePageWrapper( $text)
    {
        return '<li class="active"><span>' . $text . '</span></li>';
    }

    /**
     * 生成省略号按钮
     *
     * @return
     */
    protected function getDots()
    {
        return $this->getDisabledTextWrapper('...');
    }

    /**
     * 批量生成页码按钮.
     *
     * @param  array $urls
     * @return
     */
    protected function getUrlLinks(array $urls)
    {
        $html = '';
        foreach ($urls as $page => $url) {
            $html .= $this->getPageLinkWrapper($url, $page);
        }
        return $html;
    }

    /**
     * 生成普通页码按钮
     *
     * @param   $url
     * @param      $page
     * @return
     */
    protected function getPageLinkWrapper( $url,  $page)
    {
        if ($this->currentPage() == $page) {
            return $this->getActivePageWrapper($page);
        }
        return $this->getAvailablePageWrapper($url, $page);
    }
}
