<?php

namespace ltcms\response;

/**
 * 响应类
 */
abstract class Response
{
    /**
     * 原始数据
     * @var mixed
     */
    public $data;   //插件中间件需要修改这个值,实现动态增减参数,protected=>public

    /**
     * 模板路径
     * @var mixed
     */
    protected $templatePath;

    /**
     * 当前contentType
     * @var string
     */
    protected $contentType = 'text/html';

    /**
     * 字符集
     * @var string
     */
    protected $charset = 'utf-8';

    /**
     * 状态码
     * @var integer
     */
    protected $code = 200;

    /**
     * 输出参数
     * @var array
     */
    protected $options = [];

    /**
     * header参数
     * @var array
     */
    protected $header = [];

    /**
     * 输出内容
     * @var string
     */
    protected $content = null;

    /**
     * 初始化
     * @access protected
     * @param mixed $data 输出数据
     * @param int $code 状态码
     */
    protected function init($data = '', int $code = 200)
    {
        $this->data($data);
        $this->code = $code;

        $this->contentType($this->contentType, $this->charset);
    }

    /**
     * 创建Response对象
     * @access public
     * @param mixed $data 输出数据
     * @param string $type 输出类型
     * @param int $code 状态码
     */
    public static function create($data = '', string $type = 'html', int $code = 200)
    {
        $class = false !== strpos($type, '\\') ? $type : '\\ltcms\\response\\' . ucfirst(strtolower($type));
        return new $class($data, $code);
    }

    /**
     * 设置响应头
     * @access public
     * @param array $header 参数
     * @return $this
     */
    public function header(array $header = [])
    {
        $this->header = array_merge($this->header, $header);

        return $this;
    }

    /**
     * 输出的参数
     * @access public
     * @param mixed $options 输出参数
     * @return $this
     */
    public function options(array $options = [])
    {
        $this->options = array_merge($this->options, $options);

        return $this;
    }

    /**
     * 发送数据到客户端
     * @access public
     * @return void
     * @throws \InvalidArgumentException
     */
    public function send(): void
    {
        // 页面缓存
        ob_start();
        if (PHP_VERSION > 8.0) {
            ob_implicit_flush(false);
        } else {
            ob_implicit_flush(0);
        }
        try {
            $data = $this->getContent();
        } catch (\Exception $e) {
            throw $e;
        }
        ob_clean();

        if (!headers_sent()) {
            if (!empty($this->header)) {
                // 发送状态码
                http_response_code($this->code);
                // 发送头部信息
                foreach ($this->header as $name => $val) {
                    header($name . (!is_null($val) ? ':' . $val : ''));
                }
            }
        }

        $this->sendData($data);

        if (function_exists('fastcgi_finish_request')) {
            // 提高页面响应
            fastcgi_finish_request();
        }
    }

    /**
     * 获取输出数据
     * @access public
     * @return string
     */
    public function getContent(): string
    {

        if (null == $this->content) {
            $content = $this->output($this->data, $this->templatePath);
            if (null !== $content && !is_string($content) && !is_numeric($content) && !is_callable([$content, '__toString',])) {
                throw new \InvalidArgumentException(sprintf('variable type error： %s', gettype($content)));
            }

            $this->content = (string)$content;
        }

        return $this->content;
    }

    /**
     * 获取头部信息
     * @access public
     * @param string $name 头部名称
     * @return mixed
     */
    public function getHeader(string $name = '')
    {
        if (!empty($name)) {
            return $this->header[$name] ?? null;
        }

        return $this->header;
    }

    /**
     * 处理数据
     * @access protected
     * @param mixed $data 要处理的数据
     * @return mixed
     */
    protected function output($data, $templatePath)
    {
        return $data;
    }

    /**
     * 输出数据
     * @access protected
     * @param string $data 要处理的数据
     * @return void
     */
    protected function sendData(string $data): void
    {
        echo $data;
    }

    /**
     * 页面缓存控制
     * @access public
     * @param string $cache 状态码
     * @return $this
     */
    public function cacheControl(string $cache)
    {
        $this->header['Cache-control'] = $cache;

        return $this;
    }

    /**
     * 页面输出类型
     * @access public
     * @param string $contentType 输出类型
     * @param string $charset 输出编码
     * @return $this
     */
    public function contentType(string $contentType, string $charset = 'utf-8')
    {
        $this->header['Content-Type'] = $contentType . '; charset=' . $charset;

        return $this;
    }

    /**
     * 输出数据设置
     * @access public
     * @param mixed $data 输出数据
     * @return $this
     */
    public function data($data)
    {
        $this->data = $data;

        return $this;
    }

    /**
     * 输出数据设置模板路径
     * @access public
     * @param mixed $data 输出数据前缀
     * @return $this
     */
    public function templatePath($templatePath = "")
    {
        $this->templatePath = $templatePath;

        return $this;
    }
    /**
     * 获取响应状态码
     * @access public
     * @return int
     */
    public function getCode(): int
    {
        return $this->code;
    }
}
